/////////////////////////////////////////////////////////////////////////////////
//
// License Creative Commons Attribution-NonCommercial-ShareAlike 3.0 Unported License.
//
/////////////////////////////////////////////////////////////////////////////////
//
// This shader is based on Huw Bowles'  FSS,  Fast Adaptive Stationary Sampling,
// shader downloaded from ShaderToy.com and is therfore subject  to  ShaderToy's
// licence conditions, see above. I have slightly modified the shader by
//
//   1) Adapting it to run with the VGHD software.
//
//   2) Replacing the texture based noise function with a purely
//      procedual noise function taken from Alexander Alekseev's
//      Seascape shader, also obtained from ShaderToy.com.
//
//   3) Tidied up the code in a couple of places.
//
// If this shader is used twice, once to provide a background of  clouds  behind
// a VGHD clip and a second time to provide the foreground then  the  result  is
// not satisfactory as there is a sharp straight line across the  the  performer
// in the clip which and this does not look realistic.
//
// The effect can be mitigated by placing the clip right at the  bottom  of  the
// screen (or even just below it) obviating the need for a foreground,  but this
// is a poor solution and not very flexible. 
//
/////////////////////////////////////////////////////////////////////////////////

// Original obtained from ShaderToy.com
// Adapted, trivialy, for VGHD by TheEmu.

uniform float u_Elapsed;    // The elapsed time in seconds
uniform vec2  u_WindowSize; // Window dimensions in pixels

// Use defines here rather than edit the body of the code.

#define iGlobalTime u_Elapsed
#define iResolution u_WindowSize

/////////////////////////////////////////////////////////////////////////////////

// FASS - Fast Adaptive Stationary Sampling

// IQ made the original scene: https://www.shadertoy.com/view/XslGRr

// The problem this shader attempts to solve is to efficiently sample each pixel ray,
// keeping the samples close to stationary in world space, while adapting
// the sample rate to meet arbitrary requirements (such as concentrating samples
// near the viewer). This requires a way to smoothly interpolate between sample
// densities which is achieved as follows.

// Samples are taken at integral power of 2 increments

// Dens 2:  X               X
// Dens 4:  X       X       X       X
// Dens 8:  X   X   X   X   X   X   X   X

// If we start at density 2, we can always move down to density 4, as there is always
// a valid sample location in level 4 for every sample position in level 2.
// However from density 4, we can only move up to 2 at even sample locations

// To make the transitions between sample levels smooth, each odd sample is given a weight.
// When moving from density 4 to density 2, the odd samples will gradually be faded out.

// For a work in progress shader illustrating the sampling see
// https://www.shadertoy.com/view/XlfGzM

// The old version of this algorithm that computed samples on a binary tree is archived here
// https://www.shadertoy.com/view/XlXGz7
// My previous shaders either computed explicit intersections with the tree (slow), or divided the
// ray into chunks and efficiently computed intersections with branches of the tree based on the
// average density in each chunk. The chunk size forced a minimum sampling rate. In contrast, this
// latest version of the algorithm evaluates the desired density at each sample and has no minimum
// sampling rate, is simpler, has less local variables and holds samples COMPLETELY stationary in Z.

#define SAMPLE_COUNT 32
#define DIST_MAX 128.
#define SAMPLES_ADAPTIVITY .2
#define EPS 0.001

// cam moving in a straight line

vec3 lookDir = vec3(-1.,0.,0.);
vec3 camVel  = vec3(-1.,0.,0.);

// cam spin around on spot
//vec3 lookDir = normalize(vec3(cos(iGlobalTime),0.,sin(iGlobalTime)));
//vec3 camVel = vec3(0.);
float samplesCurvature = 0.8; // mix between fixed z and fixed radius sampling

vec3 sundir = normalize(vec3(-1.0,0.0,1.0));

// ----------------------------------------------------------------------------

// Original texture based noise function replaced by a purely  procedural
// noise function taken from Alexander Alekseev's Seascape  shader.  This
// is neccessary with the current version of VGHD as  it  seems  able  to
// only provide a single texture as input and using the clipSprite as the
// noise texture led to visible artifacts in the final scene. TheEmu.

float hash( vec2 p ) {
    float h = dot(p,vec2(127.1,311.7));
    return fract(sin(h)*43758.5453123);
}

float noise( in vec3 p ) {
    vec2 i = floor( p.xz );
    vec2 f = fract( p.xz );
    vec2 u = f*f*(3.0-2.0*f);
    return -1.0+2.0*mix( mix( hash( i + vec2(0.0,0.0) ),
                     hash( i + vec2(1.0,0.0) ), u.x),
                mix( hash( i + vec2(0.0,1.0) ),
                     hash( i + vec2(1.0,1.0) ), u.x), u.y);
}

// ----------------------------------------------------------------------------

vec4 map( in vec3 p )
{
    float d = 0.2 - p.y;

    // huwb - im moving the camera in the world instead of moving the clouds
    vec3 q = p;// - vec3(1.0,0.1,0.0)*iGlobalTime;

    float f  = 0.5000*noise( q ); q = q*2.02;
             + 0.2500*noise( q ); q = q*2.03;
             + 0.1250*noise( q ); q = q*2.01;
             + 0.0625*noise( q );

    d += 3.0 * f;

    d = clamp( d, 0.0, 1.0 );

    vec4 res = vec4( d );

    res.xyz = mix( 1.15*vec3(1.0,0.95,0.8), vec3(0.7,0.7,0.7), res.x );

    return res;
}

// ----------------------------------------------------------------------------

float spacing(float t )
{
    t = max(t,0.0);

    // unnorm pdf - plot this in graphtoy to see shape
    float pdf = 1. / (SAMPLES_ADAPTIVITY*t + 1.);

    // integral of pdf over dist
    float norm = (1.0 / SAMPLES_ADAPTIVITY)*log(1.0 + SAMPLES_ADAPTIVITY*DIST_MAX);

    // norm pdf
    pdf /= norm;

    // sample spacing for our sample count
    return 1.0 / (float(SAMPLE_COUNT) * pdf);
}

// ----------------------------------------------------------------------------

void firstT( out float t, out float dt, out float wt, out bool even )
{
    dt = exp2(floor(log2(spacing(0.0))));
    t = dt - mod(t,dt);
    even = mod(t,2.0*dt) < EPS;
    wt = 1.;
}

// ----------------------------------------------------------------------------

void nextT( inout float t, inout float dt, inout float wt, inout bool even )
{
    float s = spacing(t); // get desired sample spacing

    if( s < dt )
      { // Immediately move to higher density
        dt /= 2.0;
        even = true;
      }
    else if( even && s > 2.0*dt )
      { // Move to lower density if a sample is there
        dt *= 2.;
        wt = 1.;
        even = mod(t,2.0*dt) < EPS;
      }

    if( even )
     { // Update sample based on how far we are into its band
      wt = clamp ( 2.0 - s/dt, 0.0, 1.0 );
     }

    // next sample

    t += dt;
    even = !even;

}

// ----------------------------------------------------------------------------

float sampleWt( float wt, bool even )
{
    return even ? (2.-wt) : wt;
}

// ----------------------------------------------------------------------------

vec4 raymarch( in vec3 ro, in vec3 rd )
{
    vec4 sum = vec4(0.0);

    // Setup sampling

    float t, dt, wt; bool even;
    firstT( t, dt, wt, even );

    for(int i=0; i<SAMPLE_COUNT; i++)
    {
        if( sum.a > 0.99 ) continue;

        vec3 pos = ro + t*rd;
        vec4 col = map( pos );

        // Iq's goodness

        float dif = clamp((col.w - map(pos+0.3*sundir).w)/0.6, 0.0, 1.0 );
        vec3  lin = vec3(0.65,0.68,0.7)*1.35 + 0.45*vec3(0.7, 0.5, 0.3)*dif;

        col.xyz *= lin;
        col.a   *= 0.35;
        col.rgb *= col.a;

        // Fade samples at far field
        // .3 is an ugly fudge factor due to oversampling

        float fadeout = clamp ( 1.0 - (t/(DIST_MAX*.3)-.85)/.15,0.0,1.0);

        // Integrate

        float thisDt = dt * sampleWt( wt, even); // blend in dts
        thisDt = sqrt(thisDt/5.0 )*5.0;          // hack to soften and brighten

        sum += thisDt * col * (1.0 - sum.a) * fadeout;

        // Next sample

        nextT( t, dt, wt, even );
    }

    sum.xyz /= (0.001+sum.w);

    return clamp( sum, 0.0, 1.0 );
}

// ----------------------------------------------------------------------------

void main(void)
{
    // Rescale and re-center.

    vec2 p = 2.0 * gl_FragCoord.xy/iResolution.xy - 1.0;
   
    // "Camera" for the cloud generation.
   
    vec3 ro = vec3(0.,1.9,0.) + iGlobalTime*camVel;
   
    vec3 ww = normalize(lookDir);
    vec3 uu = normalize(cross( vec3(0.0,1.0,0.0), ww ));
    vec3 vv = normalize(cross(ww,uu));
   
    // Coefficient tweaked to improve the look of the clouds. TheEmu.
    vec3 rd = normalize( p.x*uu + 2.0*p.y*vv + 1.9*ww );
   
    // Generate the clouds using Huw Bowles's FSS method.
   
    vec4 res = raymarch( ro, rd/mix(dot(rd,ww),1.,samplesCurvature) );
    res = clamp ( res, 0.0, 1.0 );

    float sun = clamp( dot(sundir,rd), 0.0, 1.0 );
    vec3  col = vec3(0.6,0.71,0.75) - rd.y*0.2*vec3(1.0,0.5,1.0) + 0.15*0.5;

    col += 0.2 * vec3(1.0,0.6,0.1) * pow(sun,8.0);
    col *= 0.95;

    col = mix( col, res.xyz, res.w );
    col += 0.1 * vec3(1.0,0.4,0.2) * pow(sun,3.0);

    col = clamp ( col, 0.0, 1.0 );

    gl_FragColor = vec4(col,1.0);

}

// ----------------------------------------------------------------------------
